/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */

using System;
using System.Collections.Generic;

namespace org.owasp.appsensor
{
    public enum TimeIncrement
    {
        Second,
        Minute,
        Hour,
        Day
    }

    /**
     * This class is a simple container for services (by default request URIs).  
     * The monitor simply Contains information about whether or not the given
     * service is active or not, and if it is not active, when it wil be 
     * reactivated.  
     * 
     * @author Michael Coates (michael.coates .at. owasp.org) 
     *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
     * @author John Melton (jtmelton .at. gmail.com)
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since February 24, 2010
     */
    public class AppSensorServiceMonitor
    {

        /** The service considered for this service monitor, set in constructor */
        private readonly String service;

        /** The user id associated with this service - if there is one */
        private readonly String userId;

        /** whether or not the service is active, defaults to active */
        private bool active = true;

        /** time to reactivate */
        private DateTime reActivateTime = DateTime.MinValue;

        /** number used to represent a permanent lock */
        public const int PERMANENT_LOCK_DURATION = -1;

        /** Constant for second time duration */
        public const String SECOND = "s";

        /** Constant for minute time duration */
        public const String MINUTE = "m";

        /** Constant for hour time duration */
        public const String HOUR = "h";

        /** Constant for day time duration */
        public const String DAY = "d";


        /** Create a new monitor for the specified service */
        public AppSensorServiceMonitor(String service, String userId)
        {
            this.service = service;
            this.userId = userId;
        }

        /**
         * This method disables the service represented by this monitor
         * for the specified duration and time scale.  For instance
         * if the duration is "30", and the timescale is "m", the current
         * service would be disabled for 30 minutes.
         * 
         * @param duration represents unit of measure for how long 
         * 			to disable the current service, combined with timeScale
         * @param timeScale represents the scale of time to disable the 
         * 			current service, combined with duration 
         */
        public void Disable(int duration, TimeIncrement timeScale)
        {
            DateTime cal = DateTime.Now;

            if (duration < 0)
            {
                //negative duration
                throw new ArgumentException("A negative duration is not an acceptable timespan.");
            }

            //handle indefinite request first 
            if (duration == PERMANENT_LOCK_DURATION)
            {
                cal.AddYears(1000);	//adding 1,000 years - same as indefinite
            }
            else
            {
                //handle adding duration based on requested scale type
                switch ( timeScale )
                {
                    case TimeIncrement.Second:
                        cal.AddSeconds(duration);
                        break;

                    case TimeIncrement.Minute:
                        cal.AddMinutes(duration);
                        break;

                    case TimeIncrement.Hour:
                        cal.AddHours(duration);
                        break;

                    case TimeIncrement.Day:
                        cal.AddDays(duration);
                        break;
                }
            }

            //go ahead and set reactivate time after doing calendar manipulation
            this.reActivateTime = cal;

            //set this service to disabled
            this.active = false;
        }

        /**
         * Determine whether or not this service is active
         * @return true if service is active, false if not
         */
        public bool IsActive
        {
            get
            {
                return this.active;
            }
        }

        /**
         * Get time service will be re-activated
         * @return time service will be re-activated
         */
        public DateTime ReActivateTime
        {
            get
            {
                return this.reActivateTime;
            }
        }

        /**
         * Enable this service
         */
        public void Enable()
        {
            this.active = true;
            this.reActivateTime = DateTime.MinValue;
        }

        /**
         * Get the name of the service being monitored
         * @return the name of the service being monitored
         */
        public String Service
        {
            get
            {
                return this.service;
            }
        }

        /**
         * Get the user id for whom this service belongs (may be null)
         * @return user id
         */
        public String UserId
        {
            get
            {
                return this.userId;
            }
        }
    }
}